/**
 *  @file    MessageWindowPopup_patch
 *  @author  moca
 *  @version 1.0.1 2022-07-19
 */

/*:ja
 * @plugindesc MessageWindowPopup.jsとFTKR_ExMessageWindow2.jsを併用した際に起きる問題を修正します。
 * @author moca
 * @help ※このプラグインはMessageWindowPopup.jsより下に配置してください
 * 
 * このプラグインはMessageWindowPopup.jsとFTKR_ExMessageWindow2.jsを併用した際に起きる問題を修正します。
 * 
 * 
 * フキダシウィンドウの位置・サイズ調整が個別にできない問題を修正します。
 * これに伴い、フキダシウィンドウ調整のプラグインコマンドにウィンドウIDが必要になります。
 * 
 * MWP_ADJUST [設定内容] or
 * フキダシウィンドウ調整 [設定内容]
 * 　フキダシウィンドウの表示位置やサイズを微調整します。設定内容に以下を入力。
 *
 * 　POS or 位置 [X座標] [Y座標] [ウィンドウID]
 * 　　ウィンドウのX座標とY座標を調整します。指定するのは元の座標からの相対です。
 *
 * 　SIZE or サイズ [横幅] [高さ] [ウィンドウID]
 * 　　ウィンドウの横幅と高さを調整します。指定するのは元のサイズからの相対です。
 * 
 * 
 * プラグインコマンドと文章の表示を別のイベントに配置した際に正しく動作しない問題を修正します。
 * 以下のようにコモンイベント内に設定関連のプラグインコマンドをまとめて使い回すことができます。
 * 
 * ウィンドウ1表示(コモンイベント)
 * ◆プラグインコマンド：EMW_メッセージウィンドウ指定 1 終了禁止
 * 
 * ◆コモンイベント：ウィンドウ1表示
 * ◆文章：なし, ウィンドウ, 下
 * ：　　：こうするとメッセージウィンドウID1は閉じないぞ
 * ◆プラグインコマンド：EMW_メッセージウィンドウ強制終了 すべて
 * 
 * 
 * ## バージョン履歴
 * 2022/07/19 1.0.1 プラグインコマンドと文章の表示を別のイベントにしても正しく動くように
 * 2022/07/12 1.0.0 初版
 * 
 * @param KeepWindowPosition
 * @desc フキダシウィンドウの位置をウシーンが変わっても保持するか
 * @default false
 * @type boolean
 * 
 * @param KeepWindowSize
 * @desc フキダシウィンドウのサイズをシーンが変わっても保持するか
 * @default false
 * @type boolean
*/

(function() {
    'use strict';

    var getCommandName = function(command) {
        return (command || '').toUpperCase();
    };

    var getArgNumber = function(arg, min, max) {
        if (arguments.length <= 2) min = -Infinity;
        if (arguments.length <= 3) max = Infinity;
        return (parseInt(convertEscapeCharacters(arg)) || 0).clamp(min, max);
    };

    var convertEscapeCharacters = function(text) {
        if (text == null) text = '';
        var window = SceneManager._scene._windowLayer.children[0];
        return window ? window.convertEscapeCharacters(text) : text;
    };

    //FTKR_ExMessageWindow2.jsを使用しているか
    var imported_FTKR_EMW = function() {
        return typeof Imported !== 'undefined' && Imported.FTKR_EMW;
    };

    // プラグインパラメータ
    const pluginName = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
    const parameters = PluginManager.parameters(pluginName);
    const paramKeepWindowPosition = parameters['KeepWindowPosition'];
    const paramKeepWindowSize = parameters['KeepWindowSize'];

    // MessageWindowPopup.jsのプラグインパラメータ
    // TODO: 英語対応(NoUseTail)
    const paramNoUseTail = PluginManager.parameters("MessageWindowPopup")['テールを使わない'];

    // FTKR_ExMessageWindow2.js がある場合のみパッチを当てる
    if(imported_FTKR_EMW()) {

        // ===================================
        // MessageWindowPopup.js側の修正
        // ===================================

        const _Game_Interpreter_plguinCommandMessageWindowPopup = Game_Interpreter.prototype.pluginCommandMessageWindowPopup;
        Game_Interpreter.prototype.pluginCommandMessageWindowPopup = function(command, args) {
            // フキダシウィンドウ調整コマンドだけ分岐する
            switch(getCommandName(command)){
                case 'MWP_INVALID':
                case 'フキダシウィンドウ無効化':
                    $gameSystem.clearMessagePopupFree();
                    $gameTemp._windowId = 0;
                    if (imported_FTKR_EMW() && args[0]) {
                        var windowId2 = getArgNumber(args[0]);
                        if (windowId2 >= 0) $gameSystem.clearMessagePopupEx(windowId2);
                    } else {
                        $gameSystem.clearMessagePopup();
                    }
                    return;
                case 'MWP_ADJUST':
                case 'フキダシウィンドウ調整':
                    switch (getCommandName(args[0])) {
                        case 'SIZE' :
                        case 'サイズ':
                            $gameSystem.setPopupAdjustSizeEx(getArgNumber(args[1]), getArgNumber(args[2]), getArgNumber(args[3]));
                            break;
                        case 'POS' :
                        case '位置':
                            $gameSystem.setPopupAdjustPositionEx(getArgNumber(args[1]), getArgNumber(args[2]), getArgNumber(args[3]));
                            break;
                    }
                    return;
                default:
                    break;
            }
    
            _Game_Interpreter_plguinCommandMessageWindowPopup.apply(this, arguments);
        }
    
        // フキダシウィンドウのオフセットをウィンドウID毎に保持
        const _EMW_Game_System_initialize = Game_System.prototype.initialize;
        Game_System.prototype.initialize = function() {
            _EMW_Game_System_initialize.call(this);
            this._messagePopupAdjustPositions = [];
            this._messagePopupAdjustSizes = [];
        }

        Game_System.prototype.setPopupAdjustPositionEx = function(x, y, windowId) {
            this._messagePopupAdjustPositions[windowId] = [x, y];
        };

        Game_System.prototype.setPopupAdjustSizeEx = function(w, h, windowId) {
            this._messagePopupAdjustSizes[windowId] = [w, h];
        };

        Game_System.prototype.getPopupAdjustPositionEx = function(windowId) {
            return this._messagePopupAdjustPositions[windowId] !== undefined ? this._messagePopupAdjustPositions[windowId] : [0, 0];
        };

        Game_System.prototype.getPopupAdjustSizeEx = function(windowId) {
            return this._messagePopupAdjustSizes[windowId] !== undefined ? this._messagePopupAdjustSizes[windowId] : [0, 0];
        };

        // @override
        Window_Base.prototype.setPopupBasePosition = function() {
            // thisのウィンドウIDをもとにオフセットを適用
            var pos = $gameSystem.getPopupAdjustPositionEx(this._windowId);
            this.x  = this.getPopupBaseX() - this.width / 2 + (pos ? pos[0] : 0);
            this.y  = this.getPopupBaseY() - this.height - this.getHeightForPopup() + (pos ? pos[1] : 0);
        }

        // @override
        Window_MessageEx.prototype.processVirtual = function() {
            var virtual      = {};
            virtual.index    = 0;
            virtual.text     = this.convertEscapeCharacters(this._gameMessage.allText());
            virtual.maxWidth = 0;
            this.newPage(virtual);
            while (!this.isEndOfText(virtual)) {
                this.processVirtualCharacter(virtual);
            }
            virtual.y += virtual.height;
            this._subWindowY = virtual.y;
            var choices      = this._gameMessage.choices();
            if (choices && $gameSystem.getPopupSubWindowPosition() === 2) {
                virtual.y += choices.length * this._choiceWindow.lineHeight();
                virtual.maxWidth = Math.max(virtual.maxWidth, this.newLineX() + this._choiceWindow.maxChoiceWidth());
            }
            var digit = this._gameMessage.numInputMaxDigits();
            if (digit && $gameSystem.getPopupSubWindowPosition() === 2) {
                virtual.y += this._numberWindow.lineHeight();
            }
            var width  = virtual.maxWidth + this.padding * 2;
            var height = Math.max(this.getFaceHeight(), virtual.y) + this.padding * 2;
            // 変更ここから
            var adjust = $gameSystem.getPopupAdjustSizeEx(this._windowId);
            // 変更ここまで
            if (adjust) {
                width += adjust[0];
                height += adjust[1];
            }
            if (this.isUsePauseSignTextEnd()) {
                width += this._windowPauseSignSprite.width;
            } else if (paramNoUseTail) {
                height += 8;
            }
            this.width  = width;
            this.height = height;
            this.resetFontSettings();
        };

        // シーン移動時に全てのフキダシウィンドウのオフセットをリセット
        const _EMW_Scene_Map_terminate = Scene_Map.prototype.terminate;
        Scene_Map.prototype.terminate = function() {
            _EMW_Scene_Map_terminate.call(this);

            if(paramKeepWindowPosition !== true) {
                this._messagePopupAdjustPositions = [];
            }
            if(paramKeepWindowSize !== true) {
                this._messagePopupAdjustSizes = [];
            }
        }

        // ===================================
        // FTKR_ExMessageWindow2.js側の修正
        // ===================================

        const _EMW_Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
        Game_Interpreter.prototype.pluginCommand = function (command, args) {
            // メッセージウィンドウリセット時にwindowIdを0にする
            if (command.match(/EMW_(.+)/i)){
                command = RegExp.$1;
                if(matchTextToRegs(command, [/メッセージウィンドウ(.+)/, /MessageWindow_(.+)/i])){
                    command = (RegExp.$1 + '').toUpperCase();
                    switch(command) {
                        case 'リセット':
                        case 'RESET':
                        case '強制クローズ':
                        case '強制終了':
                        case 'CLOSE':
                            // this._windowId = 0;
                            $gameTemp._windowId = 0;
                            break;
                        default:
                            break;
                    }
                }
            }

            _EMW_Game_Interpreter_pluginCommand.apply(this, arguments);
        };

        const _EMW_Game_Interpreter_setMessageWindowId = Game_Interpreter.prototype.setMessageWindowId;
        Game_Interpreter.prototype.setMessageWindowId = function(args) {
            var windowId = Number(args[0] || 0);
            $gameTemp._windowId = windowId;

            _EMW_Game_Interpreter_setMessageWindowId.apply(this, arguments);
        }

        const _EMW_Game_Temp_initialize = Game_Temp.prototype.initialize;
        Game_Temp.prototype.initialize = function() {
            _EMW_Game_Temp_initialize.call(this);
            this._windowId = 0;
        }

        // 対象のウィンドウIDを保持する
        Game_Temp.prototype.EMWwindowId = function() {
            return this._windowId;
        };

        // @override
        Game_Interpreter.prototype.windowId = function() {
            return $gameTemp.EMWwindowId();
        }
    }
})();